<?php

/*
  Application Developed By Abhinav Software
  Website: http://abhinavsoftware.com
  Email: contact@abhinavsoftware.com
  Developer: Ankur Gupta (ankurgupta555@gmail.com)
  Copyright Ankur Gupta

  For licensing and terms of use please read license.txt file
 */

abstract class AbstractMenuitem extends CFormModel {

    /** @var string */
    public $title;

    /** @var string */
    public $target;

    /** @var string */
    public $type;

    /** @var integer */
    public $id;

    /** @var integer */
    public $menu_id;

    /** @var integer */
    public $pid;

    /** @var CWidget */
    public $form;

    /**
     * Returns Validation rule   
     * @return array 
     */
    public function rules() {
  
        return array(
            array("target, type, pid", "required"),
            array("title", "length", "max" => 255, "allowEmpty" => true),
            array("pid", "validate_pid"),
            array("id", "length", "allowEmpty" => true),
        );
    }

    /**
     * Returns attribute labels
     * @return array
     */
    public function attributeLabels() {
        return array(
            "title" => "Title",
            "target" => "Target",
            "type" => "Type",
            "id" => "Id",
            "pid" => "Parent"
        );
    }

    /**
     * Validation rule to check if a menu item has been set as its own subitem
     * @param type $a
     * @param type $b
     */
    public function validate_pid($a, $b) {
        if ($this->id > 0) {
            if ($this->id == $this->pid) {
                $this->addError("pid", "You can not set any menuitem as its subitem");
            }
        }
    }

    /**
     * Displays the menu settings form in admin area
     * @param Menuitem $model
     * @param string $action 
     */
    public function displayForm(Menuitem $model = null, $action = "") {
        $this->type = get_called_class();
        $this->pid = 0;

        $tree = Menuitem::model()->getTree($this->menu_id, 0, "―", 0);
        $tree[0] = "[None]";

        if (!is_null($model)) {
            $data = unserialize($model->mitem_data);
            foreach ($data as $key => $value) {
                $this->$key = $value;
            }
            $this->title = $model->mitem_title;
            $this->target = $model->mitem_target;
            $this->id = $model->mitem_id;
            $this->type = $model->mitem_type;
            $this->menu_id = $model->m_id;
            $this->pid = $model->mitem_pid;

            //hiding itself from being displayed on pid drop down menu
            unset($tree[$this->id]);

            //hiding its children from being displayed on pid drop down menu
            $children = Menuitem::model()->getTree($this->menu_id, $this->id);
            foreach ($children as $k => $v) {
                unset($tree[$k]);
            }
        }

        $this->form = Yii::app()->controller->beginWidget('CActiveForm', array(
            'id' => 'formMenuitem',
            'htmlOptions' => array('class' => 'dialog-form'),
            'action' => $action,
            'errorMessageCssClass' => "error",
        ));
        echo "<div class='form-group'>";
        echo $this->form->label($this, "title");
        echo $this->form->textField($this, "title", array("class" => "form-control"));
        echo $this->form->error($this, "title");
        echo "</div>";

        echo "<div class='form-group'>";
        echo $this->form->label($this, "pid");
        echo $this->form->dropDownlist($this, "pid", $tree, array("class" => "form-control"));
        echo $this->form->error($this, "pid");
        echo "</div>";

        echo "<div class='form-group'>";
        echo $this->form->label($this, "target");
        echo $this->form->dropDownlist($this, "target", array("_self" => "Same Window", "_blank" => "New Window"), array("class" => "form-control"));
        echo $this->form->error($this, "target");
        echo "</div>";

        $this->adminForm();

        echo "<div class='form-group'>";
        echo $this->form->hiddenField($this, "id");
        echo $this->form->hiddenField($this, "type");
        echo $this->form->hiddenField($this, "menu_id");
        echo CHtml::hiddenField("mode", "save");
        echo "</div>";

        echo "<div>";
        echo Chtml::submitButton("Save", array("class" => "btn btn-primary", "id" => "btnSubmitWidgetForm", "name" => "btnSubmitWidgetForm"));
        echo "</div>";
        Yii::app()->controller->endWidget();
    }

    /**
     * Saves the contents of settings from in database
     */
    public function saveForm() {
        $model = new Menuitem();

        if ($this->id > 0) {
            $model = Menuitem::model()->findByPk($this->id);
        }

        $class_vars = get_class_vars($this->type);
        $data = array();
        foreach ($class_vars as $k => $v) {
            $data[$k] = $this->$k;
        }
        $model->mitem_data = serialize($data);

        if ($this->id == 0 || is_null($this->id) || $this->id == "") {
            //on insert
            $model->mitem_order = Menuitem::model()->getMaxOrderIndex($this->pid) + 1;
            $model->m_id = $this->menu_id;
        } else {
            //on update
            if ($model->mitem_pid != $this->pid) {
                //if user has changed the parent id
                //1. Update the orders of sibling items.
                $sql = "UPDATE menuitems SET mitem_order = mitem_order-1 WHERE mitem_pid = :mitem_pid AND mitem_order>:mitem_order";
                $cmd = Yii::app()->db->createCommand($sql);
                $cmd->execute(array(":mitem_pid" => $model->mitem_pid, ":mitem_order" => $model->mitem_order));
                //2. Update the order of selected item
                $nextorder = Menuitem::model()->getMaxOrderIndex($this->pid) + 1;
                $model->mitem_order = $nextorder;
            }
        }
        $model->mitem_pid = $this->pid;
        $model->mitem_title = $this->title;
        $model->mitem_target = $this->target;
        $model->mitem_type = $this->type;
        if ($model->validate()) {
            $model->save();
            return true;
        }

        return $model->getErrors();
    }

    /**
     * Deletes the menu items and its subitems
     * @param Menuitem $model
     */
    public static function delete(Menuitem $model) {
        //find sub cats
        $subs = $model->getRelated("rSubMenuitem");

        //fire command to delete subs
        if (count($subs) > 0) {
            foreach ($subs as $sub) {
                AbstractMenuitem::delete($sub);
            }
        }

        //update all records order by -1 who have order greator than 
        $sql = "UPDATE menuitems SET mitem_order = mitem_order-1 WHERE mitem_order > :mitem_order AND mitem_pid = :mitem_pid";
        $cmd = Yii::app()->db->createCommand($sql);
        $cmd->execute(array(":mitem_order" => $model->mitem_order, ":mitem_pid" => $model->mitem_pid));

        //delete itself
        $model->delete();
    }

    /**
     * Displays the admin form
     */
    public abstract function adminForm();

    /**
     * Returns link (url)
     * @param array $arg
     * @param array $data
     * @return string 
     */
    public abstract function getLink($arg, $data);
}
